/**
* \file: demoscene.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Simple scene for demo application.
*
* \component: osgStream
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <osgDB/ReadFile>

#include "demoscene.h"

DemoScene::DemoScene(std::string str)
{
    init(str);
}

#define RAND(min, max) ((min) + (float)rand()/(RAND_MAX) * ((max)-(min)))

void DemoScene::init(std::string str)
{
    std::string path;

    path.append(str);

    osg::Node* obj1 = createObject(path,
                                   osg::Vec4(1.0f, 0.2f, 0.2f, 1.0f));
    path.clear();
    path.append(str);
    path.append(".0,0,180.rot");
    osg::Node* obj2 = createObject(path,
                                   osg::Vec4(0.2f, 0.2f, 1.0f, 1.0f));

    path.clear();
    path.append(str);
    path.append(".0,0,90.rot");
    osg::Node* air_obj2 = createObject(path,
                                       osg::Vec4(0.2f, 0.2f, 1.0f, 1.0f));

    for (unsigned int i=0; i < 1; ++i)
    {
        osg::Vec3 center1 (RAND(-100.0f, 100.0f), RAND(-100.0f, 100.0f), 0.0f);
//        addChild(createStaticNode(center1, obj1));

        osg::Vec3 center2 (RAND(-100, 100), RAND(-100, 100), 0.0f);
        addChild(createStaticNode(center2, obj2));

    }

    for (unsigned int i = 0; i < 1; ++i)
    {
        osg::Vec3 center(RAND(-50,50), RAND(-50, 50), RAND(10, 100));
        addChild(createAnimateNode(center,
                                   RAND(10.0, 50.0),
                                   10.0f,
                                   air_obj2));
    }
}

osg::AnimationPathCallback* DemoScene::createAnimationCallbackPath(float radius, float time)
{
    osg::ref_ptr<osg::AnimationPath> path = new osg::AnimationPath;
    path->setLoopMode(osg::AnimationPath::LOOP);

    unsigned int numSamples = 32;
    float delta_yaw = 2.0f * osg::PI / ((float) numSamples - 1.0f);
    float delta_time = time / (float) numSamples;

    for (unsigned int i = 0; i < numSamples; i++)
    {
        float yaw = delta_yaw * (float) i;
        osg::Vec3 pos (sinf(yaw) * radius, cosf(yaw) * radius, 0.0f);
        osg::Quat rot(-yaw, osg::Z_AXIS);
        path->insert(delta_time * (float)i,
                     osg::AnimationPath::ControlPoint(pos, rot));
    }

    osg::ref_ptr<osg::AnimationPathCallback> apcb =
            new osg::AnimationPathCallback;
    apcb->setAnimationPath(path.get());

    return apcb.release();
}

osg::Node* DemoScene::createObject(const std::string& filename, const osg::Vec4& color)
{
    osg::ref_ptr<osg::Node> model_node = osgDB::readNodeFile(filename);

    osg::ref_ptr<osg::Group> obj_node = new osg::Group;
    obj_node->addChild(model_node.get());

    return obj_node.release();
}

osg::MatrixTransform* DemoScene::createStaticNode(const osg::Vec3& center, osg::Node* child)
{
    osg::ref_ptr<osg::MatrixTransform> trans_node = new osg::MatrixTransform;
    trans_node->setMatrix(osg::Matrix::translate(center));
    trans_node->addChild(child);

    return trans_node.release();
}

osg::MatrixTransform* DemoScene::createAnimateNode(const osg::Vec3& center,
                                        float radius,
                                        float time,
                                        osg::Node * child)
{
    osg::ref_ptr<osg::MatrixTransform> anim_node = new osg::MatrixTransform;
    anim_node->addUpdateCallback(createAnimationCallbackPath(radius, time));
    anim_node->addChild(child);

    osg::ref_ptr<osg::MatrixTransform> trans_node = new osg::MatrixTransform;
    trans_node->setMatrix(osg::Matrix::translate(center));
    trans_node->addChild(anim_node.get());

    return trans_node.release();
}
